package com.sap.idm.vds.connector.samplefile.impl;

import java.util.Vector;
import java.util.regex.Pattern;

import com.sap.idm.vds.connector.samplefile.impl.GenericOperations;
import com.sap.idm.vds.filter.MVDFilter;
import com.sap.idm.vds.filter.MVDFilterComplex;
import com.sap.idm.vds.filter.MVDFilterSimple;
import com.sap.idm.vds.filter.MVDFilterUtilities;


/**
 * This class is an example of how a complex filter can be used. In the method main we will test how
 * the class works with testing set of attributes and filters
 * @author I054742
 *
 */
public class AcceptFilter {

	/**
     * Checks if a set of attributes matches a filter
     * @param names -> Attribute names
     * @param values -> Attribute values
     * @param filter -> Filter
     * @return -> true if matches,
     * 			  false otherwise
     */
    public boolean accept (String filter, String []names, String []values) {
    	
    	MVDFilter mFilter = MVDFilterUtilities.constructFilter(filter);
    	return this.toFilter(mFilter,names,values);
    }
    
    
    /**
     * Checks if some attribute of a set of attributes matches a filter. The filter will be checked against names and values
     * @param filter -> The filter
     * @param names -> The attribute names
     * @param values -> The attribute values
     * @return -> true if some attribute matches the filter,
     * 			  false otherwise
     */
    private boolean internAccept (String filter, String []names, String []values) {
			
    	/* Prepares the filter cleaning unnecessary trailing and leading chars */
    	filter=GenericOperations.cleanTrailingAndLeadingSpaces(filter);
    	filter = GenericOperations.cleanParenthesis(filter);
    	filter=GenericOperations.cleanTrailingAndLeadingSpaces(filter);
    	if (filter.length()==0) filter="*=*";
    	/* Splits the filter */
		String []filterSplitted = filter.split("=");
		if (filterSplitted.length<2) return false;
		String filterName = filterSplitted[0];
		String filterValue = filterSplitted[1];
		for (int i=2; i<filterSplitted.length; i++) filterValue+="="+filterSplitted[i];
		
		/* Checks if some attribute passes the filter */
		for (int i=0; i<names.length; i++) {
			if (this.stringFilter(names[i],filterName) && this.stringFilter(values[i], filterValue)) {
				return true;
			}
		}
		
		return false;
	}
    
    
    /**
     * Checks if the string 'str' matches with the regular expression 'filter'
     * @param str -> The string to be checked against the regular expression 'filter'
     * @param filter -> The regular expression
     * @return -> True if 'str' matches with the regular expression 'filter',
     * 			  false otherwise
     */
    private boolean stringFilter (String str, String filter) {
    	
    	/* Compile a pattern to be replaced in the filter */
    	Pattern pattern = Pattern.compile("\\*");
    	/* Replace in filter all the occurrences of '*' by '.*' in order to prepare 
    	 * the correct syntax for the matching of the string against the regular expression */
    	filter=pattern.matcher(filter).replaceAll(".*");
    	/* Checks if the string matches with the regular expression */
    	return Pattern.matches(filter,str);
    }
    
    
    /**
     * Checks if a filter matches a set of attributes
     * @param aMVDFilter -> Filter
     * @param names -> Attribute names
     * @param values -> Attribute values
     * @return -> true if matches,
     * 			  false otherwise  
     */
	private boolean toFilter(MVDFilter aMVDFilter, String []names, String []values) {
		/* Checks if the filter is complex or simple and calls to the correct method */
		if(aMVDFilter instanceof MVDFilterSimple)
			return this.toFilter((MVDFilterSimple)aMVDFilter,names,values);

		return this.toFilter((MVDFilterComplex)aMVDFilter,names,values);
	}

	
	/**
	 * Checks if a simple filter matches a set of attributes
	 * @param aMVDFilter -> The filter
	 * @param names -> The attribute names
	 * @param values -> The attribute values
	 * @return -> true if matches,
     * 			  false otherwise  
	 */
	private boolean toFilter(MVDFilterSimple aMVDFilter, String []names, String []values) {
		
		String opstr = "";
		String value = null;  
		/* Gets the attribute name */
		String mAttrname=aMVDFilter.getAttrName();
		/* Gets the attribute value */
		String mAttrvalue=aMVDFilter.getAttrValue();
		/* Gets the operation type */
		int mOptype=aMVDFilter.getOpType();	     	     
		
		/* Checks if the all the attributes cleaned from the filter */
		if (mAttrname.equals(MVDFilter.MVD_CLEANED)) return false;
		/* Checks if the all the attributes ignored */
		if (mAttrname.equals(MVDFilter.MVD_IGNORED)) return true;
			    
		/* For null attributes we assume that that means that any value would be good */
		if (mAttrvalue == null) value = "*";
		else value = mAttrvalue;
		
		/* If we receives a invalid operator type it will be considered as EQUAL type */
		if ((mOptype == MVDFilter.OP_EQUAL) ||  (mOptype == MVDFilter.FILTER_OP_INVALID)) {
		    opstr = "=";
		}
		/* Operator greater than */
		else if (mOptype == MVDFilter.OP_GT) {
		    opstr = ">";
		}
		/* Operator present. It will considered as EQUAL operator */
		else if (mOptype == MVDFilter.OP_PRESENT) {
		    opstr = "=";
		}
		/* Operator less than */
		else if (mOptype == MVDFilter.OP_LT) {
		    opstr = "<";
		}
		/* The substring operator will be considered as EQUAL operator */
		else if (mOptype == MVDFilter.OP_SUBSTRING) {
		    opstr = "=";
		}
		/* The filter with the expected syntax is constructed */
		String filter = "(" + mAttrname + opstr + value + ")";
		/* Checks if the transformed simple filter is accepted */
		return this.internAccept(filter,names,values);
	}

	
	/**
	 * Checks if a complex filter matches a set of attributes
	 * @param aMVDFilter -> The filter
	 * @param names -> The attribute names
	 * @param values -> The attribute values
	 * @return -> true if matches,
     * 			  false otherwise  
	 */
	private boolean toFilter(MVDFilterComplex aMVDFilter, String []names, String []values) {
			 	
		/* It will hold the temporary final result */
		boolean tmp;
		/* Gets the operator type of this complex filter */
		int mFilterOpType=aMVDFilter.getFilterOpType();
		/* Gets the set of associated filters to this operator */
	    Vector mFilters=aMVDFilter.getFilters();
		/* Takes the first associated filter */
        MVDFilter mOne = (MVDFilter) mFilters.elementAt(0);
        if (mOne != null) {
        	/* For NOT operator there will be only an associated filter */
            if (mFilterOpType == MVDFilter.FILTER_OP_NOT) {
            	if(mOne instanceof MVDFilterComplex)
            		tmp = toFilter((MVDFilterComplex)mOne,names,values);
            	else {
            		tmp = toFilter((MVDFilterSimple)mOne,names,values);
                }
                return !tmp;
            }
            /* For AND and OR operations at least two associated filters are necessary */
            else if (mFilters.size() > 1) {
            	/* For AND operator we will check the complete set of associated filter. If we find
            	 * some associated filter with false value then the result will be false. True otherwise */
                if (mFilterOpType == MVDFilter.FILTER_OP_AND) {
                    for (int x = 0; x <  mFilters.size(); x++) {
                    	if(mFilters.elementAt(x) instanceof MVDFilterComplex) {
                    		tmp = toFilter((MVDFilterComplex)mFilters.elementAt(x),names,values);
                    		if(tmp==false) return false;
                    	} else {
                    		tmp = toFilter((MVDFilterSimple)mFilters.elementAt(x),names,values);
                        	if(tmp==false) return false;
                        }
                    }
                   return true;
                }
                else if (mFilterOpType == MVDFilter.FILTER_OP_OR) {
                	/* For OR operator we will check the complete set of associated filter. If we find
                	 * some associated filter with true value then the result will be true. False otherwise */
                    for (int x = 0; x <  mFilters.size(); x++) {
                    	if(mFilters.elementAt(x) instanceof MVDFilterComplex) {
                    		tmp =   toFilter((MVDFilterComplex) mFilters.elementAt(x),names,values);
                    		if (tmp) return true;
                    	} else {
                    		tmp = toFilter((MVDFilterSimple) mFilters.elementAt(x),names,values);
                    		if (tmp) return true;
                    	}
                    }
                    return false;
                }
            }
            else {
                // --- there should be at least two of partial filters
            }
        }
        return true;
	}
	
	
	static public void main (String []args) {
		
		/* Set of attribute names */
		String []names = {"firstname",
						  "surname",
						  "middelname",
						  "personal email",
						  "company email",
						  "address",
						  "fix phone",
						  "mobile phone",
						  "fax"
						  };
		/* Set of attribute values */
		String []values = {"emilio",
						   "garcia tormo",
						   "jose",
						   "emilio@gemilio.com",
						   "numberemilio@company.com",
						   "downtown street, 20 A",
						   "+47 12345678",
						   "+47 22335588",
						   "+47 12333321"
						   };
		/* Set of filters */
		String []filters = {"(*name*=emilio)",
							"(surname=emilio)",
							"(*phone=*12345678)",
							"(email=emilio*)",
							"(*email*=emilio*)",
							"*=*20*",
							"(&(*name*=emilio)(*phone=*8)(*name*=jose))",
							"(&(*name*=emilio)(*phone=*7))",
							"(|(*name*=emilio)(*phone=*7))",
							"(!(*name*=emilio))",
							"(!(firstname=emilio))",
							"(!(firstname=adsf))"
							};
		
		AcceptFilter af = new AcceptFilter();
		/* Shows the list of attributes */
		System.out.println("List of attributes\n--------------------");
		for (int i=0; i<names.length; i++) {
			System.out.println(names[i]+ " --> " + values[i]);
		}
		/* Shows the results of filtering */
		System.out.println("\nResults of filtering\n-------------------");
		for (int i=0; i<filters.length; i++) {
			System.out.println(filters[i]+(af.accept(filters[i],names,values)?" is":" is NOT")+" accepted");
		}
	}
}
